-- tracker_deaths.lua (IMPROVED with Fatality-style detection + Server Translation)
-- WhoDAT - Enhanced Death Tracking for Wrath 3.3.5a
-- 
-- MULTI-SERVER SUPPORT:
-- This tracker uses the shared server_translator.lua module to normalize combat log
-- events from different servers (Warmane, etc.) to retail format. The tracking code
-- works with normalized data and doesn't need server-specific logic.
--
-- Supported servers (via server_translator.lua):
--   - Retail WotLK 3.3.5 (standard format)
--   - Warmane (translated from scrambled format)
--   - Easy to add more! See server_translator.lua
--
-- See SERVER_TRANSLATOR_USAGE.md for integration details.

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}
_G[ADDON_NAME] = NS

-- ============================================================================
-- 3.3.5-Compatible Timer System (C_Timer doesn't exist in WotLK)
-- ============================================================================

local timerFrame = CreateFrame("Frame")
local activeTimers = {}

local function After(delay, callback)
  local timer = {
    endTime = GetTime() + delay,
    callback = callback,
  }
  table.insert(activeTimers, timer)
  
  -- Start the update loop if not already running
  if not timerFrame:GetScript("OnUpdate") then
    timerFrame:SetScript("OnUpdate", function(self, elapsed)
      local now = GetTime()
      for i = #activeTimers, 1, -1 do
        if now >= activeTimers[i].endTime then
          activeTimers[i].callback()
          table.remove(activeTimers, i)
        end
      end
      
      -- Stop update loop when no timers
      if #activeTimers == 0 then
        self:SetScript("OnUpdate", nil)
      end
    end)
  end
end

-- ============================================================================
-- Constants & Upvalues
-- ============================================================================

local bit_band, bit_bor = bit.band, bit.bor
local GetTime, UnitGUID, UnitName, UnitExists = GetTime, UnitGUID, UnitName, UnitExists
local UnitHealth, UnitHealthMax, UnitIsDead = UnitHealth, UnitHealthMax, UnitIsDead
local UnitBuff, GetNumRaidMembers, GetNumPartyMembers = UnitBuff, GetNumRaidMembers, GetNumPartyMembers

-- Warmane-safe flag helpers
local WD_AssumeUnknownDestIsPlayer = true

-- Combat log constants (cached defensively)
local CL_AFFIL_MINE   = _G.COMBATLOG_OBJECT_AFFILIATION_MINE
local CL_AFFIL_PARTY  = _G.COMBATLOG_OBJECT_AFFILIATION_PARTY
local CL_AFFIL_RAID   = _G.COMBATLOG_OBJECT_AFFILIATION_RAID
local CL_TYPE_PLAYER  = _G.COMBATLOG_OBJECT_TYPE_PLAYER
local CL_TYPE_PET     = _G.COMBATLOG_OBJECT_TYPE_PET
local CL_TYPE_NPC     = _G.COMBATLOG_OBJECT_TYPE_NPC
local CL_REACTION_FRIENDLY = _G.COMBATLOG_OBJECT_REACTION_FRIENDLY
local CL_RAIDTARGET1  = _G.COMBATLOG_OBJECT_RAIDTARGET1
local CL_SPECIAL_MASK = _G.COMBATLOG_OBJECT_SPECIAL_MASK

-- Spirit of Redemption spell ID
local SPIRIT_OF_REDEMPTION = GetSpellInfo(27827)

-- Timeouts and limits
local ATTACKER_TIMEOUT = 10  -- seconds to keep attackers in memory
local EVENT_HISTORY_SIZE = 10  -- number of recent damage events to track per target
local CLEANUP_INTERVAL = 5  -- seconds between cleanup runs

-- ============================================================================
-- Fatality-style Damage Event Tracking
-- ============================================================================

-- candidates[destGUID] = array of recent damage events
-- Each event: {time, srcGUID, srcName, srcFlags, destGUID, destName, destFlags,
--              spellID, spellName, amount, overkill, crit, crush, environment}
local candidates = {}
local playerGUID = nil

-- Special event types that can kill
local special_damage_events = {
  SPELL_DAMAGE = true,
  SPELL_PERIODIC_DAMAGE = true,
  RANGE_DAMAGE = true,
  SWING_DAMAGE = true,
  DAMAGE_SPLIT = true,
  DAMAGE_SHIELD = true,
}

local environmental_damage_events = {
  ENVIRONMENTAL_DAMAGE = true,
}

-- ============================================================================
-- Server Translation
-- ============================================================================
-- Combat log translation is handled by the shared server_translator.lua module.
-- It auto-detects the server type and normalizes all combat log events to retail format.
-- See NS.ServerTranslator API in server_translator.lua

-- ============================================================================
-- Helper Functions
-- ============================================================================

local function ToNumberFlags(flags)
  if flags == nil then return 0 end
  local t = type(flags)
  if t == "number" then return flags end
  if t == "string" then
    local n = tonumber(flags) or tonumber(flags, 16)
    return n or 0
  end
  return 0
end

local function Band(a, b)
  return bit_band(ToNumberFlags(a), ToNumberFlags(b))
end

local function ObjectFlagsToType(flags)
  local f = ToNumberFlags(flags)
  if CL_TYPE_PLAYER and Band(f, CL_TYPE_PLAYER) > 0 then return "player" end
  if CL_TYPE_PET    and Band(f, CL_TYPE_PET)    > 0 then return "pet"    end
  if CL_TYPE_NPC    and Band(f, CL_TYPE_NPC)    > 0 then return "npc"    end
  return "unknown"
end

-- Warmane name normalization (sourceGUID is often the actual name)
local function NormalizeSourceName(sourceName, sourceGUID)
  -- Priority 1: Try sourceName first if it's a valid player-like name
  if type(sourceName) == "string" and sourceName ~= "" and sourceName ~= "Unknown" then
    -- Check if it looks like a player name (has letters, reasonable length, no special patterns)
    if sourceName:find("%a") and #sourceName <= 20 and not sourceName:find("^0x") and not sourceName:find("Creature") then
      return sourceName
    end
  end
  
  -- Priority 2: Try sourceGUID only for special Warmane cases where GUID actually contains a name
  -- But exclude obvious GUIDs (long alphanumeric strings, hex patterns)
  if type(sourceGUID) == "string" and sourceGUID ~= "" and sourceGUID:find("%a") then
    -- Only use GUID if it looks like a name (short, no hex patterns, no colons)
    if #sourceGUID <= 20 and not sourceGUID:find("^0x") and not sourceGUID:find(":") and not sourceGUID:find("%d%d%d%d") then
      return sourceGUID
    end
  end
  
  -- Priority 3: Handle numeric sourceName (likely a creature ID)
  if type(sourceName) == "number" then
    if UnitExists("target") then
      local tName = UnitName("target")
      if tName and tName ~= "" then
        return tName
      end
    end
    return ("Creature #%d"):format(sourceName)
  end
  
  -- Priority 4: If sourceName exists but looks weird, still try to use it
  if type(sourceName) == "string" and sourceName ~= "" then
    return sourceName
  end
  
  -- Fallback - guaranteed to return a string
  return "Unknown"
end

-- Check if a unit is the player or in the player's group
local function IsPlayerOrGroup(name)
  if not name or name == "" then return false end
  local playerName = UnitName("player")
  if name == playerName then return true end
  
  -- Check raid
  if GetNumRaidMembers() > 0 then
    for i = 1, 40 do
      if UnitName("raid"..i) == name then
        return true
      end
    end
  end
  
  -- Check party
  if GetNumPartyMembers() > 0 then
    for i = 1, 4 do
      if UnitName("party"..i) == name then
        return true
      end
    end
  end
  
  return false
end

-- Determine if flags indicate a friendly NPC (for Warmane quirk handling)
local function IsFriendlyNPC(sourceFlags, sourceName)
  local sourceType = ObjectFlagsToType(sourceFlags)
  
  -- If already marked as NPC, keep it
  if sourceType == "npc" then return true end
  
  -- Warmane quirk: friendly NPCs sometimes have player flags
  if sourceType == "player" and not IsPlayerOrGroup(sourceName) then
    if CL_REACTION_FRIENDLY and Band(sourceFlags, CL_REACTION_FRIENDLY) > 0 then
      return true
    end
  end
  
  return false
end

-- ============================================================================
-- Fatality-style Damage Event Recording
-- ============================================================================

local function RecordDamageEvent(timestamp, srcGUID, srcName, srcFlags, destGUID, destName, destFlags,
                                  spellID, spellName, spellSchool, amount, overkill, school, resisted,
                                  blocked, absorbed, critical, glancing, crushing, isEnvironmental)
  
  -- Only track damage to the player
  if not destGUID then return end
  
  -- Initialize candidate list for this destination if needed
  if not candidates[destGUID] then
    candidates[destGUID] = {}
  end
  
  local events = candidates[destGUID]
  
  -- Normalize source name
  local normalizedSrcName = NormalizeSourceName(srcName, srcGUID)
  
  -- Determine actual source type (accounting for Warmane quirks)
  local sourceType = ObjectFlagsToType(srcFlags)
  if IsFriendlyNPC(srcFlags, normalizedSrcName) then
    sourceType = "npc"
  end
  
  -- Create event record
  local event = {
    time        = timestamp or GetTime(),
    srcGUID     = srcGUID,
    srcName     = normalizedSrcName,
    srcFlags    = srcFlags,
    destGUID    = destGUID,
    destName    = destName,
    destFlags   = destFlags,
    spellID     = spellID,
    spellName   = spellName,
    amount      = amount or 0,
    overkill    = overkill or 0,
    crit        = critical and true or false,
    crush       = crushing and true or false,
    environment = isEnvironmental and true or false,
    sourceType  = sourceType,
  }
  
  -- Add to the end of the array
  table.insert(events, event)
  
  NS.Log("DEBUG", "Recorded damage event: %s hit %s for %d (total events: %d)", 
    event.srcName or "Unknown", destName or "Unknown", event.amount, #events)
  
  -- Keep only the most recent events (rolling buffer)
  while #events > EVENT_HISTORY_SIZE do
    table.remove(events, 1)  -- Remove oldest
  end
end

-- ============================================================================
-- State Tracking (Original System - for fallback)
-- ============================================================================

local deathState = {
  -- Multi-source attacker tracking (fallback system)
  recentAttackers   = {},
  lastDamageEvent   = nil,
  
  -- Environmental tracking
  activeDebuffs     = {},
  waterBreathing    = 0,
  
  -- Combat tracking
  combatStartTime   = nil,
  inCombat          = false,
  
  -- Death tracking
  durabilityBeforeDeath = nil,
  deathTime         = nil,
  deathClock        = nil,
  pendingDeath      = nil,
  
  -- Killing blow tracking
  killingBlow       = nil,
}

local function RecordAttackerFallback(sourceGUID, sourceName, sourceFlags, damageAmount, overkill, spellName)
  if not sourceGUID and not sourceName then return end
  local now = GetTime()
  local normName = NormalizeSourceName(sourceName, sourceGUID)
  local sourceType = ObjectFlagsToType(sourceFlags)
  
  if IsFriendlyNPC(sourceFlags, normName) then
    sourceType = "npc"
  end
  
  local attacker = deathState.recentAttackers[normName]
  if not attacker then
    attacker = {
      name        = normName,
      type        = sourceType,
      guid        = sourceGUID,
      flags       = sourceFlags,
      totalDamage = 0,
      hits        = 0,
      spells      = {},
      lastHit     = 0,
      lastDamage  = 0,
    }
    deathState.recentAttackers[normName] = attacker
  end
  
  attacker.totalDamage = (attacker.totalDamage or 0) + (damageAmount or 0)
  attacker.hits        = (attacker.hits or 0) + 1
  attacker.lastHit     = now
  attacker.lastDamage  = damageAmount or 0
  attacker.lastOverkill = overkill or 0
  
  if spellName then
    attacker.spells[spellName] = (attacker.spells[spellName] or 0) + 1
  end
end

-- ============================================================================
-- Debug: Combat Log Event Monitoring
-- ============================================================================

local combatLogMonitoring = false
local monitoredEventCount = 0

local function MonitorCombatLog(timestamp, event, hideCaster, srcGUID, srcName, srcFlags, srcRaidFlags,
                                 destGUID, destName, destFlags, destRaidFlags, ...)
  if not combatLogMonitoring then return end
  
  monitoredEventCount = monitoredEventCount + 1
  
  local playerName = UnitName("player")
  local pGUID = UnitGUID("player")
  
  -- Log events involving the player
  if destName == playerName or destGUID == pGUID or srcName == playerName or srcGUID == pGUID then
    print(string.format("[CL Monitor #%d] %s: src=%s/%s dest=%s/%s", 
      monitoredEventCount, event, 
      tostring(srcName), tostring(srcGUID),
      tostring(destName), tostring(destGUID)))
  end
end

-- ============================================================================
-- Combat Log Processing
-- ============================================================================

local function OnCombatLog(...)
  
  -- TRANSLATE COMBAT LOG FOR SERVER COMPATIBILITY
  -- Uses shared server_translator.lua module to normalize Warmane/other servers to retail format
  -- IMPORTANT: Must capture ALL return values, not just first 11!
  local timestamp, event, hideCaster, srcGUID, srcName, srcFlags, srcRaidFlags,
        destGUID, destName, destFlags, destRaidFlags,
        arg12, arg13, arg14, arg15, arg16, arg17, arg18, arg19, arg20, arg21, arg22 =
    NS.ServerTranslator:TranslateCombatLog(...)
  
  -- Debug monitoring (now with translated params)
  MonitorCombatLog(timestamp, event, hideCaster, srcGUID, srcName, srcFlags, srcRaidFlags,
                   destGUID, destName, destFlags, destRaidFlags,
                   arg12, arg13, arg14, arg15, arg16, arg17, arg18, arg19, arg20, arg21, arg22)
  
  -- Ensure playerGUID is set
  if not playerGUID then
    playerGUID = UnitGUID("player")
    -- On non-retail servers, playerGUID might need translation
    if playerGUID then
      playerGUID = NS.ServerTranslator:TranslateGUID(playerGUID, UnitName("player"))
    end
    if playerGUID then
      NS.Log("DEBUG", "PlayerGUID initialized in combat log: %s", tostring(playerGUID))
    end
  end
  
  -- Debug: Log all events that target the player
  if destGUID == playerGUID or destName == UnitName("player") then
    NS.Log("DEBUG", "Combat log event for player: %s from %s (destGUID match: %s)", 
      event, srcName or "Unknown", tostring(destGUID == playerGUID))
  end
  
  -- Only process if destination is the player
  if destGUID ~= playerGUID then
    -- Server fallback: check by name if GUID doesn't match
    local playerName = UnitName("player")
    if destName ~= playerName and not (destGUID and destGUID:match(":" .. playerName .. "$")) then
      return
    end
    -- If name matches but GUID doesn't, update our playerGUID
    NS.Log("WARN", "PlayerGUID mismatch - updating from %s to %s", tostring(playerGUID), tostring(destGUID))
    playerGUID = destGUID
  end
  
  -- Handle damage events
  if special_damage_events[event] then
    local spellID, spellName, spellSchool, amount, overkill, school, resisted, blocked, absorbed, critical, glancing, crushing
    
    if event == "SWING_DAMAGE" then
      -- SWING_DAMAGE: amount is at position 12, etc.
      amount, overkill, school, resisted, blocked, absorbed, critical, glancing, crushing =
        arg12, arg13, arg14, arg15, arg16, arg17, arg18, arg19, arg20
      spellName = "Melee"
    else
      -- SPELL_DAMAGE/etc: spellID at 12, spellName at 13, amount at 15, etc.
      spellID, spellName, spellSchool, amount, overkill, school, resisted, blocked, absorbed, critical, glancing, crushing =
        arg12, arg13, arg14, arg15, arg16, arg17, arg18, arg19, arg20, arg21, arg22
    end
    
    -- Record in Fatality-style candidate system
    RecordDamageEvent(timestamp, srcGUID, srcName, srcFlags, destGUID, destName, destFlags,
                      spellID, spellName, spellSchool, amount, overkill, school, resisted,
                      blocked, absorbed, critical, glancing, crushing, false)
    
    -- Also record in fallback attacker system
    RecordAttackerFallback(srcGUID, srcName, srcFlags, amount, overkill, spellName)
    
    -- Track as potential killing blow if it has overkill
    if overkill and overkill > 0 then
      deathState.killingBlow = {
        name      = NormalizeSourceName(srcName, srcGUID),
        type      = ObjectFlagsToType(srcFlags),
        spell     = spellName,
        damage    = amount,
        overkill  = overkill,
        timestamp = timestamp or GetTime(),
      }
    end
    
    deathState.lastDamageEvent = GetTime()
    
  elseif environmental_damage_events[event] then
    -- Environmental damage: environmentalType at position 12, amount at 13, etc.
    local environmentalType, amount, overkill, school, resisted, blocked, absorbed, critical, glancing, crushing =
      arg12, arg13, arg14, arg15, arg16, arg17, arg18, arg19, arg20, arg21
    
    local envName = environmentalType or "Environment"
    
    -- Record in Fatality-style system
    RecordDamageEvent(timestamp, nil, envName, 0, destGUID, destName, destFlags,
                      nil, envName, nil, amount, overkill, school, resisted,
                      blocked, absorbed, critical, glancing, crushing, true)
    
    -- Record in fallback system
    RecordAttackerFallback(nil, envName, 0, amount, overkill, envName)
    
    if overkill and overkill > 0 then
      deathState.killingBlow = {
        name      = envName,
        type      = "environment",
        spell     = envName,
        damage    = amount,
        overkill  = overkill,
        timestamp = timestamp or GetTime(),
      }
    end
    
    deathState.lastDamageEvent = GetTime()
  end
  
  -- Track debuffs for environmental deaths
  if event == "SPELL_AURA_APPLIED" or event == "SPELL_AURA_REFRESH" then
    local spellID, spellName = arg12, arg13
    if spellName then
      deathState.activeDebuffs[spellName] = GetTime()
    end
  elseif event == "SPELL_AURA_REMOVED" then
    local spellID, spellName = arg12, arg13
    if spellName then
      deathState.activeDebuffs[spellName] = nil
    end
  end
end

-- ============================================================================
-- Fatality-style Death Detection (PRIMARY METHOD)
-- ============================================================================

local function GetKillerFromCandidates(guid)
  NS.Log("DEBUG", "GetKillerFromCandidates called with guid: %s", tostring(guid))
  
  local events = candidates[guid]
  if not events or #events == 0 then
    NS.Log("DEBUG", "No candidate events found (events: %s, count: %d)", 
      tostring(events), events and #events or 0)
    return nil, "no_candidates"
  end
  
  NS.Log("DEBUG", "Found %d candidate events", #events)
  
  -- Get the most recent damage event (killing blow)
  local killingEvent = events[#events]
  if not killingEvent then
    return nil, "no_killing_event"
  end
  
  NS.Log("DEBUG", "Last damage event: %s hit for %d with %s", 
    killingEvent.srcName or "Unknown",
    killingEvent.amount or 0,
    killingEvent.spellName or "Unknown")
  
  local killer = {
    name       = killingEvent.srcName or "Unknown",
    type       = killingEvent.sourceType or "unknown",
    spell      = killingEvent.spellName or "Unknown",
    damage     = killingEvent.amount or 0,
    overkill   = killingEvent.overkill or 0,
    guid       = killingEvent.srcGUID,
    flags      = killingEvent.srcFlags,
    timestamp  = killingEvent.time,
    confidence = "fatality_primary",
    method     = "fatality_last_damage",
  }
  
  return killer, nil
end

-- ============================================================================
-- Weighted Detection (FALLBACK METHOD)
-- ============================================================================

local function GetKillerWeighted()
  local recentAttackersList = {}
  local now = GetTime()
  
  for name, attacker in pairs(deathState.recentAttackers) do
    if (now - (attacker.lastHit or 0)) <= ATTACKER_TIMEOUT then
      table.insert(recentAttackersList, {
        name    = name,
        type    = attacker.type,
        damage  = attacker.totalDamage,
        hits    = attacker.hits,
        recency = now - (attacker.lastHit or 0),
        guid    = attacker.guid,
        flags   = attacker.flags,
      })
    end
  end
  
  if #recentAttackersList == 0 then
    return nil, "no_recent_attackers"
  end
  
  -- Sort by total damage
  table.sort(recentAttackersList, function(a, b)
    return (a.damage or 0) > (b.damage or 0)
  end)
  
  local topAttacker = recentAttackersList[1]
  
  return {
    name       = topAttacker.name,
    type       = topAttacker.type,
    damage     = topAttacker.damage,
    guid       = topAttacker.guid,
    flags      = topAttacker.flags,
    confidence = "weighted_fallback",
    method     = "weighted_damage",
  }, nil
end

-- ============================================================================
-- Environmental Detection (TERTIARY FALLBACK)
-- ============================================================================

local function GetKillerEnvironmental()
  -- Check for drowning
  local maxBreath = GetMirrorTimerInfo(2) or 0
  if deathState.waterBreathing > 0 and maxBreath > 0 then
    local remaining = GetMirrorTimerInfo(2)
    if remaining and remaining <= 0 then
      return {
        name       = "Drowning",
        type       = "environment",
        confidence = "environmental",
        method     = "breath_detection",
      }, nil
    end
  end
  
  -- Check for active DoTs
  for debuffName, timestamp in pairs(deathState.activeDebuffs) do
    if (GetTime() - timestamp) < 5 then
      return {
        name       = debuffName,
        type       = "spell_dot",
        confidence = "environmental",
        method     = "active_debuff",
      }, nil
    end
  end
  
  -- Check if last damage was environmental
  if deathState.lastDamageEvent and (GetTime() - deathState.lastDamageEvent) < 3 then
    for name, attacker in pairs(deathState.recentAttackers) do
      if attacker.type == "environment" and (GetTime() - (attacker.lastHit or 0)) < 3 then
        return {
          name       = name,
          type       = "environment",
          damage     = attacker.totalDamage,
          confidence = "environmental",
          method     = "recent_env_damage",
        }, nil
      end
    end
  end
  
  return nil, "no_environmental_cause"
end

-- ============================================================================
-- Unified Killer Detection
-- ============================================================================

local function DetermineKiller()
  NS.Log("DEBUG", "=== Determining killer ===")
  NS.Log("DEBUG", "PlayerGUID: %s", tostring(playerGUID))
  
  -- METHOD 1: Fatality-style last damage event (PRIMARY)
  local killer, err = GetKillerFromCandidates(playerGUID)
  if killer then
    NS.Log("DEBUG", "âœ“ Killer determined by Fatality method: %s", killer.name)
    return killer
  end
  NS.Log("DEBUG", "âœ— Fatality method failed: %s", err or "unknown")
  
  -- METHOD 2: Killing blow with overkill (if available)
  if deathState.killingBlow and (GetTime() - deathState.killingBlow.timestamp) < 2 then
    NS.Log("DEBUG", "âœ“ Killer determined by killing blow: %s", deathState.killingBlow.name)
    deathState.killingBlow.confidence = "killing_blow"
    deathState.killingBlow.method = "overkill_detection"
    return deathState.killingBlow
  end
  NS.Log("DEBUG", "âœ— Killing blow not available or too old")
  
  -- METHOD 3: Weighted attacker analysis (FALLBACK)
  killer, err = GetKillerWeighted()
  if killer then
    NS.Log("DEBUG", "âœ“ Killer determined by weighted method: %s", killer.name)
    return killer
  end
  NS.Log("DEBUG", "âœ— Weighted method failed: %s", err or "unknown")
  
  -- METHOD 4: Environmental detection (TERTIARY)
  killer, err = GetKillerEnvironmental()
  if killer then
    NS.Log("DEBUG", "âœ“ Killer determined by environmental method: %s", killer.name)
    return killer
  end
  NS.Log("DEBUG", "âœ— Environmental method failed: %s", err or "unknown")
  
  -- FINAL FALLBACK: Unknown
  NS.Log("WARN", "Could not determine killer - all methods failed")
  NS.Log("DEBUG", "Candidates for playerGUID: %d events", 
    (playerGUID and candidates[playerGUID] and #candidates[playerGUID]) or 0)
  NS.Log("DEBUG", "Recent attackers: %d", 
    (function() local c=0; for _ in pairs(deathState.recentAttackers) do c=c+1 end return c end)())
  
  return {
    name       = "Unknown",
    type       = "unknown",
    confidence = "failed",
    method     = "no_detection",
  }
end

-- ============================================================================
-- Death Processing
-- ============================================================================

local function GetAverageDurability()
  local sum, count = 0, 0
  for i = 1, 18 do
    local current, maximum = GetInventoryItemDurability(i)
    if current and maximum and maximum > 0 then
      sum = sum + (current / maximum * 100)
      count = count + 1
    end
  end
  return count > 0 and (sum / count) or 100
end

local function GetDurabilityLoss(before, after)
  return math.max(0, before - after)
end

local function OnPlayerDead()
  if deathState.pendingDeath then
    return  -- Already processing a death
  end
  
  local now = GetTime()
  deathState.deathTime = time()
  deathState.deathClock = now
  deathState.durabilityBeforeDeath = GetAverageDurability()
  
  -- Set pending flag IMMEDIATELY to prevent double-processing
  deathState.pendingDeath = { processing = true }
  
  -- Small delay to ensure all combat log events are processed
  After(0.3, function()
    -- Determine killer using unified detection
    local killer = DetermineKiller()
    
    -- Get location info
    local zone = GetZoneText() or "Unknown Zone"
    local subzone = GetSubZoneText() or ""
    
    -- Get map coordinates
    local mapID = GetCurrentMapAreaID() or 0
    local x, y = 0, 0
    if GetPlayerMapPosition then
      x, y = GetPlayerMapPosition("player")
    end
    
    -- Get combat duration
    local combatDuration = 0
    if deathState.combatStartTime then
      combatDuration = now - deathState.combatStartTime
    end
    
    -- Count attackers
    local attackerCount = 0
    for _ in pairs(deathState.recentAttackers) do
      attackerCount = attackerCount + 1
    end
    
    -- Build death record
    deathState.pendingDeath = {
      ts                  = deathState.deathTime,
      zone                = zone,
      subzone             = subzone,
      map_id              = mapID,
      x                   = x,
      y                   = y,
      killer_name         = killer.name,
      killer_type         = killer.type,
      killer_spell        = killer.spell,
      killer_damage       = killer.damage,
      killer_overkill     = killer.overkill,
      killer_guid         = killer.guid,
      killer_method       = killer.method,
      killer_confidence   = killer.confidence,
      combat_duration     = combatDuration,
      attacker_count      = attackerCount,
      durability_before   = deathState.durabilityBeforeDeath,
      level               = UnitLevel("player"),
    }
    
    -- Build attacker list for detailed logging
    local recentAttackersList = {}
    for name, attacker in pairs(deathState.recentAttackers) do
      if (now - (attacker.lastHit or 0)) <= ATTACKER_TIMEOUT then
        table.insert(recentAttackersList, {
          name   = name,
          type   = attacker.type,
          damage = attacker.totalDamage,
          hits   = attacker.hits,
        })
      end
    end
    table.sort(recentAttackersList, function(a, b) return a.damage > b.damage end)
    
    -- Log the death
    NS.Log("WARN", "%s killed by %s (%s) in %s%s",
      UnitName("player"), killer.name, killer.confidence or "unknown",
      zone, (subzone ~= "" and (" - " .. subzone) or "")
    )
    
    if #recentAttackersList > 1 then
      NS.Log("DEBUG", "Recent attackers:")
      for i = 1, math.min(5, #recentAttackersList) do
        local a = recentAttackersList[i]
        NS.Log("DEBUG", string.format(" %d. %s (%d damage, %d hits)", i, a.name, a.damage, a.hits))
      end
    end
  end)
end

local function OnPlayerAlive()
  if not deathState.pendingDeath then return end
  
  local deathDuration = time() - (deathState.deathTime or time())
  local durabilityAfter = GetAverageDurability()
  local durabilityLoss = GetDurabilityLoss(deathState.durabilityBeforeDeath or 100, durabilityAfter)
  
  -- Determine resurrection type
  local rezType = "spirit"
  for i = 1, 40 do
    local name = UnitBuff("player", i)
    if not name then break end
    if name == "Soulstone Resurrection" then
      rezType = "soulstone"
      break
    elseif name:match("Rebirth") or name:match("Resurrection") or name:match("Redemption") then
      rezType = "class_rez"
      break
    end
  end
  
  if rezType == "spirit" and durabilityLoss < 1 then
    rezType = "class_rez"
  elseif rezType == "spirit" and deathDuration < 10 then
    rezType = "corpse"
  end
  
  -- Complete the death record
  deathState.pendingDeath.durability_after = durabilityAfter
  deathState.pendingDeath.durability_loss = durabilityLoss
  deathState.pendingDeath.rez_type = rezType
  deathState.pendingDeath.rez_time = deathDuration
  deathState.pendingDeath.rez_ts = time()
  
  -- Save to database
  local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
              or (GetRealmName() .. ":" .. UnitName("player"))
  
  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or {}
  WhoDatDB.characters[key].events = WhoDatDB.characters[key].events or {}
  WhoDatDB.characters[key].events.deaths = WhoDatDB.characters[key].events.deaths or {}
  
  table.insert(WhoDatDB.characters[key].events.deaths, deathState.pendingDeath)
  
  -- Emit event
  if NS.EventBus and NS.EventBus.Emit then
    NS.EventBus:Emit("player", "death", deathState.pendingDeath)
  end
  
  NS.Log("INFO", "Death logged: %s (method: %s, confidence: %s, dur loss: %.1f%%, rez: %s)",
    deathState.pendingDeath.killer_name,
    deathState.pendingDeath.killer_method,
    deathState.pendingDeath.killer_confidence,
    durabilityLoss,
    rezType
  )
  
  -- Clear state
  deathState.pendingDeath = nil
  deathState.combatStartTime = nil
  deathState.recentAttackers = {}
  deathState.killingBlow = nil
  deathState.lastDamageEvent = nil
  deathState.activeDebuffs = {}
  
  -- Clear Fatality-style candidates
  if playerGUID and candidates[playerGUID] then
    candidates[playerGUID] = {}
  end
end

-- ============================================================================
-- UNIT_HEALTH Death Detection (Fatality-style)
-- ============================================================================

local function OnUnitHealth(unit)
  -- Only monitor player
  if unit ~= "player" then return end
  
  -- Check if dead or has Spirit of Redemption
  if UnitIsDead("player") or UnitBuff("player", SPIRIT_OF_REDEMPTION) then
    -- Trigger death processing if not already pending
    if not deathState.pendingDeath then
      OnPlayerDead()
    end
  end
end

-- ============================================================================
-- Cleanup
-- ============================================================================

local function CleanupOldAttackers()
  local now = GetTime()
  
  -- Clean up fallback attacker tracking
  for name, attacker in pairs(deathState.recentAttackers) do
    if (now - (attacker.lastHit or 0)) > ATTACKER_TIMEOUT then
      deathState.recentAttackers[name] = nil
    end
  end
  
  -- Clean up Fatality-style candidate events
  if playerGUID and candidates[playerGUID] then
    local events = candidates[playerGUID]
    local cutoff = now - ATTACKER_TIMEOUT
    
    -- Remove old events from the beginning
    while #events > 0 and events[1].time < cutoff do
      table.remove(events, 1)
    end
  end
end

-- ============================================================================
-- Event Registration
-- ============================================================================

local frame = CreateFrame("Frame")
frame:RegisterEvent("PLAYER_LOGIN")
frame:RegisterEvent("PLAYER_DEAD")
frame:RegisterEvent("PLAYER_ALIVE")
frame:RegisterEvent("PLAYER_UNGHOST")
frame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
frame:RegisterEvent("PLAYER_REGEN_DISABLED")
frame:RegisterEvent("PLAYER_REGEN_ENABLED")
frame:RegisterEvent("MIRROR_TIMER_START")
frame:RegisterEvent("UNIT_HEALTH")

local timeSinceLastCleanup = 0
frame:SetScript("OnUpdate", function(_, elapsed)
  timeSinceLastCleanup = timeSinceLastCleanup + elapsed
  if timeSinceLastCleanup >= CLEANUP_INTERVAL then
    CleanupOldAttackers()
    timeSinceLastCleanup = 0
  end
end)

frame:SetScript("OnEvent", function(_, event, ...)
  if event == "PLAYER_LOGIN" then
    playerGUID = UnitGUID("player")
    NS.Log("INFO", "Death tracker initialized - PlayerGUID: %s", tostring(playerGUID))
  elseif event == "PLAYER_DEAD" then
    -- Ensure GUID is set
    if not playerGUID then
      playerGUID = UnitGUID("player")
      NS.Log("WARN", "PlayerGUID was not set at death, initializing now: %s", tostring(playerGUID))
    end
    OnPlayerDead()
  elseif event == "PLAYER_ALIVE" or event == "PLAYER_UNGHOST" then
    OnPlayerAlive()
  elseif event == "COMBAT_LOG_EVENT_UNFILTERED" then
    OnCombatLog(...)
  elseif event == "PLAYER_REGEN_DISABLED" then
    deathState.combatStartTime = GetTime()
    deathState.inCombat = true
  elseif event == "PLAYER_REGEN_ENABLED" then
    deathState.inCombat = false
    if not deathState.pendingDeath then
      deathState.combatStartTime = nil
    end
  elseif event == "MIRROR_TIMER_START" then
    local timerType = select(1, ...)
    if timerType == "BREATH" then
      deathState.waterBreathing = GetMirrorTimerInfo(2)
    end
  elseif event == "UNIT_HEALTH" then
    OnUnitHealth(...)
  end
end)

-- ============================================================================
-- Debug Commands
-- ============================================================================

SLASH_WDDEATHS1 = "/wddeaths"
SlashCmdList["WDDEATHS"] = function(msg)
  msg = (msg or ""):lower()
  
  if msg == "stats" then
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
                or (GetRealmName() .. ":" .. UnitName("player"))
    local char = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key]
    local deaths = char and char.events and char.events.deaths or {}
    
    print("=== Death Statistics ===")
    print(string.format("Total deaths: %d", #deaths))
    
    if #deaths > 0 then
      local byConfidence, byMethod = {}, {}
      for _, death in ipairs(deaths) do
        local conf = death.killer_confidence or "unknown"
        local method = death.killer_method or "unknown"
        byConfidence[conf] = (byConfidence[conf] or 0) + 1
        byMethod[method] = (byMethod[method] or 0) + 1
      end
      
      print("\nKiller Detection Confidence:")
      for conf, count in pairs(byConfidence) do
        print(string.format(" %s: %d (%.1f%%)", conf, count, count / #deaths * 100))
      end
      
      print("\nDetection Methods Used:")
      for method, count in pairs(byMethod) do
        print(string.format(" %s: %d", method, count))
      end
    end
    
  elseif msg == "recent" then
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
                or (GetRealmName() .. ":" .. UnitName("player"))
    local char = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key]
    local deaths = char and char.events and char.events.deaths or {}
    
    print("=== Recent Deaths ===")
    local start = math.max(1, #deaths - 9)
    for i = start, #deaths do
      local death = deaths[i]
      local timestamp = date("%Y-%m-%d %H:%M", death.ts)
      print(string.format("[%s] %s (%s conf, %s) in %s",
        timestamp,
        death.killer_name,
        death.killer_confidence or "?",
        death.killer_method or "?",
        death.zone))
      if death.attacker_count and death.attacker_count > 1 then
        print(string.format(" + %d other attackers involved", death.attacker_count - 1))
      end
    end
    
  elseif msg == "attackers" then
    print("=== Current Attackers ===")
    local now = GetTime()
    local count = 0
    
    for _, attacker in pairs(deathState.recentAttackers) do
      local timeSince = now - (attacker.lastHit or 0)
      print(string.format("%s (%s): %d damage, %d hits, %.1fs ago",
        attacker.name, attacker.type, attacker.totalDamage, attacker.hits, timeSince))
      count = count + 1
    end
    
    if count == 0 then
      print("No recent attackers")
    end
    
  elseif msg == "candidates" then
    print("=== Fatality Candidates ===")
    if playerGUID and candidates[playerGUID] then
      local events = candidates[playerGUID]
      print(string.format("Tracking %d recent damage events:", #events))
      for i = #events, math.max(1, #events - 9), -1 do
        local event = events[i]
        local age = GetTime() - event.time
        print(string.format(" [%.1fs ago] %s: %d dmg%s from %s",
          age,
          event.srcName or "Unknown",
          event.amount or 0,
          event.overkill and event.overkill > 0 and string.format(" (+%d overkill)", event.overkill) or "",
          event.spellName or "Unknown"))
      end
    else
      print("No candidate events tracked")
      print("PlayerGUID: " .. tostring(playerGUID))
    end
  
  elseif msg == "debug" or msg == "status" then
    print("=== Death Tracker Debug Status ===")
    print(string.format("PlayerGUID: %s", tostring(playerGUID)))
    print(string.format("In combat: %s", tostring(deathState.inCombat)))
    print(string.format("Pending death: %s", tostring(deathState.pendingDeath ~= nil)))
    
    -- Candidate events
    if playerGUID and candidates[playerGUID] then
      print(string.format("Candidate events: %d", #candidates[playerGUID]))
    else
      print("Candidate events: 0 (or playerGUID not set)")
    end
    
    -- Recent attackers
    local attackerCount = 0
    for _ in pairs(deathState.recentAttackers) do
      attackerCount = attackerCount + 1
    end
    print(string.format("Recent attackers: %d", attackerCount))
    
    -- Last damage
    if deathState.lastDamageEvent then
      local age = GetTime() - deathState.lastDamageEvent
      print(string.format("Last damage: %.1fs ago", age))
    else
      print("Last damage: never")
    end
    
    -- Killing blow
    if deathState.killingBlow then
      local age = GetTime() - deathState.killingBlow.timestamp
      print(string.format("Killing blow: %s (%.1fs ago)", deathState.killingBlow.name, age))
    else
      print("Killing blow: none")
    end
  
  elseif msg == "monitor" then
    combatLogMonitoring = not combatLogMonitoring
    monitoredEventCount = 0
    if combatLogMonitoring then
      print("=== Combat Log Monitoring ENABLED ===")
      print("All combat log events involving you will be printed.")
      print("Type '/wddeaths monitor' again to disable.")
    else
      print("=== Combat Log Monitoring DISABLED ===")
    end
    
  else
    print("=== WhoDAT Enhanced Death Tracker ===")
    print("/wddeaths stats      - Show death statistics")
    print("/wddeaths recent     - Show recent deaths")
    print("/wddeaths attackers  - Show current attackers")
    print("/wddeaths candidates - Show Fatality damage event buffer")
    print("/wddeaths debug      - Show real-time tracking status")
    print("/wddeaths monitor    - Toggle combat log event monitoring")
  end
end

NS.Log("INFO", "Enhanced death tracker loaded (Fatality + Weighted detection)")

return NS