-- WhoDAT - tracker_quests.lua
-- Quests, reputation, skills, tradeskills, spellbook + glyphs, companions/pets
-- Wrath-safe; null checks for Warmane and defensive gating

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}
_G[ADDON_NAME] = NS
local U = NS.Utils or {}

-- ============================================================================
-- CONFIGURATION (NEW: Added for quest addon integration)
-- ============================================================================
local CONFIG_QUESTS = {
  -- Force built-in quest tracking even if external addons present
  FORCE_BUILTIN_TRACKING = false,
  
  -- Use external addons to enhance quest ID detection
  USE_ADDONS_FOR_QUEST_IDS = true,
  
  -- Use external addons for better objective progress tracking
  USE_ADDONS_FOR_OBJECTIVES = true,
  
  -- Addon preference order (first detected = first used)
  ADDON_PRIORITY = { "QuestHelper", "Questie" },
}

-- ============================================================================
-- EXTERNAL ADDON DETECTION (NEW)
-- ============================================================================
local activeQuestSource = nil
local externalAddonsAvailable = {}

local QUEST_ADDON_DETECTORS = {
  Questie = function()
    return _G.Questie ~= nil 
      and _G.QuestieDB ~= nil
      and _G.QuestieQuest ~= nil
  end,
  
  QuestHelper = function()
    return _G.QuestHelper ~= nil
      and type(_G.QuestHelper_Objectives) == "table"
  end,
}

local function DetectQuestAddons()
  externalAddonsAvailable = {}
  
  if CONFIG_QUESTS.FORCE_BUILTIN_TRACKING then
    activeQuestSource = "WhoDAT_builtin"
    return
  end
  
  for _, addonName in ipairs(CONFIG_QUESTS.ADDON_PRIORITY) do
    local detector = QUEST_ADDON_DETECTORS[addonName]
    if detector and detector() then
      table.insert(externalAddonsAvailable, addonName)
      if not activeQuestSource then
        activeQuestSource = addonName
      end
    end
  end
  
  if not activeQuestSource then
    activeQuestSource = "WhoDAT_builtin"
  end
end

-- ============================================================================
-- QUEST ADDON INTEGRATION HELPERS (NEW)
-- These improve data quality without changing output schema
-- ============================================================================

-- Try to get quest ID from Quest Helper (fallback when Blizzard API fails)
local function GetQuestHelperID(questTitle)
  if not CONFIG_QUESTS.USE_ADDONS_FOR_QUEST_IDS then return nil end
  if not _G.QuestHelper or not _G.QuestHelper_Objectives then return nil end
  
  for questID, data in pairs(_G.QuestHelper_Objectives) do
    if type(data) == "table" then
      local title = nil
      if type(data[1]) == "string" then
        title = data[1]
      elseif type(data.name) == "string" then
        title = data.name
      elseif type(data.title) == "string" then
        title = data.title
      end
      
      if title == questTitle then
        return tonumber(questID)
      end
    end
  end
  
  return nil
end

-- Try to get quest ID from Questie
local function GetQuestieID(questTitle)
  if not CONFIG_QUESTS.USE_ADDONS_FOR_QUEST_IDS then return nil end
  if not _G.Questie or not _G.QuestieDB or not _G.QuestieDB.questData then return nil end
  
  for questID, quest in pairs(_G.QuestieDB.questData) do
    if quest.Name == questTitle or quest.name == questTitle then
      return tonumber(questID)
    end
  end
  
  return nil
end

-- Try to get better objective progress from Quest Helper
local function GetQuestHelperObjectiveProgress(questTitle, objectiveText)
  if not CONFIG_QUESTS.USE_ADDONS_FOR_OBJECTIVES then return nil, nil end
  if not _G.QuestHelper or not _G.QuestHelper_Objectives then return nil, nil end
  
  for questID, data in pairs(_G.QuestHelper_Objectives) do
    if type(data) == "table" and type(data.objectives) == "table" then
      local title = data[1] or data.name or data.title
      if title == questTitle then
        for _, obj in ipairs(data.objectives) do
          local objText = obj.text or obj.desc or tostring(obj)
          if objText == objectiveText or objText:find(objectiveText, 1, true) then
            local current = obj.current or obj.progress or obj.have or 0
            local needed = obj.needed or obj.total or obj.require or 0
            return current, needed
          end
        end
      end
    end
  end
  
  return nil, nil
end

-- Try to get better objective progress from Questie
local function GetQuestieObjectiveProgress(questID, objectiveIndex)
  if not CONFIG_QUESTS.USE_ADDONS_FOR_OBJECTIVES then return nil, nil end
  if not _G.Questie or not _G.QuestieQuest then return nil end
  
  local ok, current = pcall(function()
    if _G.QuestieQuest.GetObjectiveProgress then
      return _G.QuestieQuest:GetObjectiveProgress(questID, objectiveIndex)
    end
  end)
  
  if ok and current then
    local quest = _G.QuestieDB and _G.QuestieDB.questData and _G.QuestieDB.questData[questID]
    if quest and quest.Objectives and quest.Objectives[objectiveIndex] then
      local needed = quest.Objectives[objectiveIndex].Needed or quest.Objectives[objectiveIndex].required
      return current, needed
    end
    return current, nil
  end
  
  return nil, nil
end
--------------------------------------------------------------------------------
-- Helpers & DB guards
--------------------------------------------------------------------------------

-- === Simple diff + safe logger ===
local function _logEvent(category, action, payload)
  -- Prefer global WhoDAT_LogEvent if present; otherwise fall back to NS.Log
  if type(WhoDAT_LogEvent) == "function" then
    WhoDAT_LogEvent(category, action, payload)
  elseif NS.Log then
    NS.Log("EVENT", "%s:%s %s",
      tostring(category),
      tostring(action),
      tostring(
        (payload and payload.spell) or
        (payload and payload.glyph) or
        (payload and payload.skill) or
        (payload and payload.talent) or
        ""
      )
    )
  end
end

-- Build name->item map from { {name=..., ...}, ... } array
local function _indexByName(arr)
  local m = {}
  if type(arr) == "table" then
    for _, v in ipairs(arr) do
      if type(v) == "table" and v.name then m[v.name] = v end
    end
  end
  return m
end

-- Generic set diff: returns added[], removed[], common[name] = {old=..., new=...}
local function _setDiff(oldArr, newArr)
  local oldM, newM = _indexByName(oldArr), _indexByName(newArr)
  local added, removed, common = {}, {}, {}

  -- added
  for name, v in pairs(newM) do
    if not oldM[name] then
      table.insert(added, v)
    else
      common[name] = { old = oldM[name], new = v }
    end
  end

  -- removed
  for name, v in pairs(oldM) do
    if not newM[name] then
      table.insert(removed, v)
    end
  end

  return added, removed, common
end

-- Extract numeric progress ("3/10") from objective text; returns cur,total or nil
local function _parseProgress(text)
  if type(text) ~= "string" then return nil end
  -- Common WotLK pattern: "Collect Foo: 3/10" or "3/10 Foo slain"
  local cur, total = text:match("(%d+)%s*/%s*(%d+)")
  if cur and total then
    return tonumber(cur), tonumber(total)
  end
  return nil
end

-- Persist "previous snapshot" store per character safely
local function _ensurePrevStore(char)
  char.prev = char.prev or {}
  char.prev.snapshots = char.prev.snapshots or {}
  char.prev.series = char.prev.series or {}  -- in case you want series diffs later
  return char.prev
end

local function now() return time() end

local function EnsureCharacterBranches()
  -- Derive player key via Utils if present; otherwise fallback to name@realm
  local key
  if U and type(U.GetPlayerKey) == "function" then
    key = U.GetPlayerKey()
  else
    local name = (type(UnitName)=="function" and UnitName("player")) or "Player"
    local realm = (type(GetRealmName)=="function" and GetRealmName()) or "Realm"
    key = string.format("%s@%s", name or "Player", realm or "Realm")
  end

  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or { events = {}, series = {}, snapshots = {} }

  local c = WhoDatDB.characters[key]
  c.events  = c.events  or {}
  c.series  = c.series  or {}
  c.snapshots = c.snapshots or {}
  return key, c
end

local function push(t, v) t[#t+1] = v end

-- Track quests seen as completed in this session, to disambiguate removal vs abandonment
local _recentlyCompleted = {} -- [questID or title] = ts
local function markCompleted(id, title) 
  if id then _recentlyCompleted[tostring(id)] = now() end
  if title then _recentlyCompleted[title] = now() end
end
local function wasRecentlyCompleted(id, title, withinSeconds)
  local t1 = id and _recentlyCompleted[tostring(id)]
  local t2 = title and _recentlyCompleted[title]
  local t = t1 or t2
  return t and (now() - t) <= (withinSeconds or 10)
end


-- ============================================================================
-- IMPROVED ABANDONMENT DETECTION
-- Track quest removals to detect rapid add/remove cycles (UI reloads)
-- ============================================================================
local _recentlyRemoved = {} -- [questID or title] = { ts = timestamp, title = string, id = mixed }
local _confirmedAbandoned = {} -- [questID or title] = ts (only emit event once per quest)

local ABANDONMENT_GRACE_PERIOD = 5 -- seconds to wait before confirming abandonment
local ABANDONMENT_DEBOUNCE = 60 -- don't emit duplicate abandonment events within 60s

local function markQuestRemoved(id, title)
  local key = tostring(id)
  _recentlyRemoved[key] = { ts = now(), title = title, id = id }
  if title then
    _recentlyRemoved[title] = { ts = now(), title = title, id = id }
  end
end

local function wasRecentlyRemoved(id, title, withinSeconds)
  local t1 = id and _recentlyRemoved[tostring(id)]
  local t2 = title and _recentlyRemoved[title]
  local info = t1 or t2
  return info and (now() - info.ts) <= (withinSeconds or ABANDONMENT_GRACE_PERIOD)
end

local function clearRecentlyRemoved(id, title)
  if id then _recentlyRemoved[tostring(id)] = nil end
  if title then _recentlyRemoved[title] = nil end
end

local function wasRecentlyAbandonedEvent(id, title)
  local t1 = id and _confirmedAbandoned[tostring(id)]
  local t2 = title and _confirmedAbandoned[title]
  local t = t1 or t2
  return t and (now() - t) <= ABANDONMENT_DEBOUNCE
end

local function markAbandonedEventEmitted(id, title)
  if id then _confirmedAbandoned[tostring(id)] = now() end
  if title then _confirmedAbandoned[title] = now() end
end

-- ============================================================================
-- QUESTHELPER / QUESTIE STATE VERIFICATION
-- Check if external addons think a quest is still active
-- ============================================================================
local function IsQuestActiveInQuestHelper(questTitle)
  if not _G.QuestHelper or not _G.QuestHelper_Objectives then return nil end
  
  for questID, data in pairs(_G.QuestHelper_Objectives) do
    if type(data) == "table" then
      local title = data[1] or data.name or data.title
      if title == questTitle then
        return true
      end
    end
  end
  
  return false
end

local function IsQuestActiveInQuestie(questID, questTitle)
  if not _G.Questie or not _G.QuestieQuest then return nil end
  
  -- Try by ID first if we have it
  if questID and type(_G.QuestieQuest.GetQuest) == "function" then
    local ok, quest = pcall(_G.QuestieQuest.GetQuest, _G.QuestieQuest, questID)
    if ok and quest then
      return true
    end
  end
  
  -- Fallback: check Questie's quest data
  if questTitle and _G.QuestieDB and _G.QuestieDB.questData then
    for qid, quest in pairs(_G.QuestieDB.questData) do
      if (quest.Name == questTitle or quest.name == questTitle) then
        if type(_G.QuestieQuest.IsQuestInLog) == "function" then
          local ok, inLog = pcall(_G.QuestieQuest.IsQuestInLog, _G.QuestieQuest, qid)
          if ok and inLog then
            return true
          end
        end
      end
    end
  end
  
  return false
end

local function VerifyQuestNotActive(id, title)
  -- If we have external addons, check with them before confirming abandonment
  if CONFIG_QUESTS.USE_ADDONS_FOR_QUEST_IDS then
    local qhActive = IsQuestActiveInQuestHelper(title)
    if qhActive then
      if NS.Log then
        NS.Log("DEBUG", "Quest '%s' still active in QuestHelper, not marking as abandoned", title or "Unknown")
      end
      return false
    end
    
    local qiActive = IsQuestActiveInQuestie(id, title)
    if qiActive then
      if NS.Log then
        NS.Log("DEBUG", "Quest '%s' still active in Questie, not marking as abandoned", title or "Unknown")
      end
      return false
    end
  end
  
  return true
end


-- Simple throttle map so repeated events don't spam scans
local _lastRun = {}
local function throttled(tag, seconds)
  local t = _lastRun[tag] or 0
  local nowTs = now()
  if nowTs - t >= seconds then
    _lastRun[tag] = nowTs
    return true
  end
  return false
end

-- Safe pcall wrapper
local function safe(fn, ...)
  local ok, err = pcall(fn, ...)
  if not ok and DEFAULT_CHAT_FRAME then
    DEFAULT_CHAT_FRAME:AddMessage("|cffff7f7f[WhoDAT]|r tracker_quests error: " .. tostring(err))
  end
  return ok
end

-- ============================================================================
-- Quest Reward Tracking State
-- ============================================================================
local questRewardPending = nil

-- ============================================================================
-- Helper: Safe Quality Normalization
-- ============================================================================
-- GetQuestItemInfo can return quality = -1 for items not yet cached.
-- This helper ensures quality is always a valid non-negative integer.
local function SafeNormalizeQuality(quality, itemLink)
  -- If quality is valid (0-6), use it
  if quality and quality >= 0 and quality <= 6 then
    return quality
  end
  
  -- Fallback 1: Try to extract quality from item link color code
  if itemLink and type(itemLink) == "string" then
    local colorCode = itemLink:match("|c(%x%x%x%x%x%x%x%x)|H")
    if colorCode then
      -- Map WoW color codes to quality values
      local colorToQuality = {
        ["ff9d9d9d"] = 0, -- Poor (gray)
        ["ffffffff"] = 1, -- Common (white)
        ["ff1eff00"] = 2, -- Uncommon (green)
        ["ff0070dd"] = 3, -- Rare (blue)
        ["ffa335ee"] = 4, -- Epic (purple)
        ["ffff8000"] = 5, -- Legendary (orange)
        ["ffe6cc80"] = 6, -- Artifact (light orange, TBC/WotLK)
        ["ffe5cc80"] = 6, -- Artifact alternate
      }
      local extractedQuality = colorToQuality[colorCode:lower()]
      if extractedQuality then
        return extractedQuality
      end
    end
  end
  
  -- Fallback 2: Default to Common (white) for unknown quality
  -- This is safer than 0 (Poor) as most quest rewards are at least Common
  return 1
end

-- ============================================================================
-- Quest Reward Capture (QUEST_COMPLETE)
-- ============================================================================
local function OnQuestCompleteDialog()
  local questTitle = GetTitleText()
  local numChoices = GetNumQuestChoices()
  local numRewards = GetNumQuestRewards()
  
  -- Try to find quest ID from quest log
  local questID = nil
  local questLevel = nil
  local numEntries = GetNumQuestLogEntries()
  for i = 1, numEntries do
    local title, level, _, isHeader, _, isComplete, qID = GetQuestLogTitle(i)
    if title == questTitle and not isHeader and qID then
      questID = qID
      questLevel = level
      break
    end
  end
  
  -- Build reward_choices array
  local rewardChoices = {}
  for i = 1, numChoices do
    local name, texture, numItems, quality, isUsable = GetQuestItemInfo("choice", i)
    local link = GetQuestItemLink("choice", i)
    if name then
      table.insert(rewardChoices, {
        name = name,
        link = link,
        quantity = numItems or 1,
        quality = SafeNormalizeQuality(quality, link),
        texture = texture,
      })
    end
  end
  
  -- Build reward_required array (always given, not a choice)
  local rewardRequired = {}
  for i = 1, numRewards do
    local name, texture, numItems, quality, isUsable = GetQuestItemInfo("reward", i)
    local link = GetQuestItemLink("reward", i)
    if name then
      table.insert(rewardRequired, {
        name = name,
        link = link,
        quantity = numItems or 1,
        quality = SafeNormalizeQuality(quality, link),
        texture = texture,
      })
    end
  end
  
  questRewardPending = {
    quest_id = questID,
    quest_title = questTitle,
    quest_level = questLevel,
    money = GetRewardMoney(),
    xp = GetRewardXP(),
    honor = (type(GetRewardHonor) == "function") and GetRewardHonor() or 0,
    arena = (type(GetRewardArenaPoints) == "function") and GetRewardArenaPoints() or 0,
    reward_choices = rewardChoices,
    reward_required = rewardRequired,
    ts = time(),
  }
end

-- ============================================================================
-- Quest Turn-In Handler (QUEST_FINISHED)
-- ============================================================================
local function OnQuestTurnedIn()
  if not questRewardPending then return end
  
  -- Add zone context
  questRewardPending.zone = GetRealZoneText() or GetZoneText()
  questRewardPending.subzone = GetSubZoneText()
  
  -- Save to database
  local key, char = EnsureCharacterBranches()
  char.events = char.events or {}
  char.events.quest_rewards = char.events.quest_rewards or {}
  
  table.insert(char.events.quest_rewards, questRewardPending)
  
  if NS.Log then
    NS.Log("INFO", "Quest reward tracked: %s (money: %dg, xp: %d)",
      questRewardPending.quest_title or "Unknown",
      math.floor((questRewardPending.money or 0) / 10000),
      questRewardPending.xp or 0)
  end
  
  questRewardPending = nil
end

--------------------------------------------------------------------------------
-- Public: Scan orchestrator
--------------------------------------------------------------------------------
function NS.Quests_ScanAll()
  if throttled("Quests_ScanAll", 3) then -- avoid running too frequently
    safe(NS.Quests_ScanLog)
    safe(NS.Quests_ScanReputation)
    safe(NS.Quests_ScanSkills)
    safe(NS.Quests_ScanSpellbook)
    safe(NS.Quests_ScanGlyphs)
    safe(NS.Quests_ScanCompanions)
    safe(NS.Quests_ScanPetStable)
    safe(NS.Quests_ScanPetInfo)
    safe(NS.Quests_ScanPetSpellbook)
    safe(NS.Quests_ScanTradeSkills)
  end
end

--------------------------------------------------------------------------------
-- Quests log (append-only event entries)
--------------------------------------------------------------------------------
function NS.Quests_ScanLog()
  if not throttled("Quests_ScanLog", 1) then return end

  local key, char = EnsureCharacterBranches()
  char.events.quests = char.events.quests or {}
  local prev = _ensurePrevStore(char)

  -- Build current compact quest snapshot for diffing
  local current = {} -- array of { id, title, complete, objectives = { {text, cur, total, complete}, ... } }
  local indexById = {} -- map for quick lookup by id

  local num = (type(GetNumQuestLogEntries)=="function" and GetNumQuestLogEntries()) or 0
  for i = 1, num do
    local title, level, _, isHeader, _, isComplete, questID = GetQuestLogTitle(i)
    if title then  -- Server may not return questID
      SelectQuestLogEntry(i)
      -- Objectives (with parsed numeric progress if available)
      local objectives = {}
      local ocount = (type(GetNumQuestLeaderBoards)=="function" and GetNumQuestLeaderBoards(i)) or 0
      
      -- Skip real headers (marked as header AND have no objectives)
      local isRealHeader = isHeader and (ocount == 0)
      
      if not isRealHeader then
      for oi = 1, ocount do
        local otext, objType, finished = GetQuestLogLeaderBoard(oi, i)
        
        -- Try to parse progress from text
        local cur, total = _parseProgress(otext)
        
        -- If parsing failed, try external addons for better progress tracking
        if not cur or not total then
          -- Try Quest Helper first
          local qhCur, qhTotal = GetQuestHelperObjectiveProgress(title, otext)
          if qhCur and qhTotal then
            cur, total = qhCur, qhTotal
          elseif questID then
            -- Try Questie as fallback
            local qiCur, qiTotal = GetQuestieObjectiveProgress(questID, oi)
            if qiCur and qiTotal then
              cur, total = qiCur, qiTotal
            end
          end
        end
        
        push(objectives, { text = otext, type = objType, complete = finished and true or false, cur = cur, total = total })
      end

        -- Use questID if available, otherwise try external addons, then fall back to index
        local questId = questID

        if not questId or questId == 0 then
          -- Try Quest Helper first (user's preferred addon)
          questId = GetQuestHelperID(title)
          
          -- Try Questie as fallback
          if not questId then
            questId = GetQuestieID(title)
          end
          
          -- Final fallback: use quest log index
          if not questId then
            questId = i
          end
        end
        
        -- Record normalized row
        local row = {
          id = questId, 
          title = title, 
          complete = isComplete and true or false, 
          objectives = objectives,
        }
        push(current, row)
        indexById[questId] = row

        -- (Optional) keep your rich append-only detail as-is
        -- ... your existing push(char.events.quests, {...}) block remains unchanged ...
      end
    end
  end

  -- Previous snapshot (compact) for diffing
  prev.snapshots = prev.snapshots or {}
  local old = prev.snapshots.quest_log or {}

  -- === Diff: added/removed/common (by quest id preferred) ===
  local function _asNameArray(arr)
    local out = {}
    for _, q in ipairs(arr or {}) do
      -- Normalize 'name' field for _setDiff to compare: prefer unique id as string
      table.insert(out, { name = tostring(q.title or q.id), _q = q })
    end
    return out
  end

  local added, removed, common = _setDiff(_asNameArray(old), _asNameArray(current))

  -- === Emit ACCEPTED for newly added quests ===
  for _, v in ipairs(added) do
    local q = v._q
    
    -- Check if this quest was recently removed (UI reload detection)
    if wasRecentlyRemoved(q.id, q.title, ABANDONMENT_GRACE_PERIOD) then
      -- Quest was just removed and re-added quickly (likely UI reload)
      clearRecentlyRemoved(q.id, q.title)
      if NS.Log then
        NS.Log("DEBUG", "Quest '%s' re-added after brief removal (UI reload), skipping duplicate events", q.title)
      end
    else
      -- This is a genuine new quest acceptance
      _logEvent("quests", "accepted", { id = q.id, title = q.title })
    end
  end

  -- === Emit COMPLETED for status change old.complete=false -> new.complete=true ===
  for _, pair in pairs(common) do
    local oldQ = pair.old._q
    local newQ = pair.new._q
    if (not oldQ.complete) and newQ.complete then
      _logEvent("quests", "completed", { id = newQ.id, title = newQ.title })
      markCompleted(newQ.id, newQ.title)
      -- Clear any pending removal tracking when quest completes
      clearRecentlyRemoved(newQ.id, newQ.title)
    end
  end

  -- === Handle removed quests (may be completed or abandoned) ===
  for _, v in ipairs(removed) do
    local oldQ = v._q
    local id = oldQ.id
    local title = oldQ.title
    
    -- First check: was it recently completed?
    if wasRecentlyCompleted(id, title, 30) then
      -- Quest completed recently, removal is expected
      clearRecentlyRemoved(id, title)
    else
      -- Mark as recently removed and verify with external addons
      markQuestRemoved(id, title)
      
      -- Don't emit abandonment immediately - wait for grace period
      -- This will be handled by a delayed check or next scan
      if NS.Log then
        NS.Log("DEBUG", "Quest '%s' removed from log, marking for verification", title or id)
      end
    end
  end
  
  -- === Verify and emit abandonments for quests that stayed removed ===
  -- Check for quests that were removed in previous scans and never came back
  local toClean = {}
  for key, info in pairs(_recentlyRemoved) do
    local elapsed = now() - info.ts
    
    -- Grace period has passed, verify abandonment
    if elapsed >= ABANDONMENT_GRACE_PERIOD then
      -- Check if it's actually abandoned or just temporarily missing
      if VerifyQuestNotActive(info.id, info.title) and 
         not wasRecentlyCompleted(info.id, info.title, 30) and
         not wasRecentlyAbandonedEvent(info.id, info.title) then
        
        -- Confirmed abandonment
        _logEvent("quests", "abandoned", { id = info.id, title = info.title })
        markAbandonedEventEmitted(info.id, info.title)
        
        if NS.Log then
          NS.Log("INFO", "Quest ABANDONED (verified): %s", info.title or info.id)
        end
      end
      
      -- Clean up this tracking entry
      table.insert(toClean, key)
    end
  end
  
  -- Clean up verified entries
  for _, key in ipairs(toClean) do
    _recentlyRemoved[key] = nil
  end

  -- === Objective progress events (diff each objective entry by text) ===
  -- Build lookup table for previous objectives by quest id+objective text
  local prevObj = {} -- key "id\ntext" -> {cur,total,complete}
  for _, oq in ipairs(old or {}) do
    local q = oq._q
    if q and q.objectives then
      for _, o in ipairs(q.objectives) do
        prevObj[(tostring(q.title) .. "\n" .. tostring(o.text))] = { cur = o.cur, total = o.total, complete = o.complete }
      end
    end
  end

  for _, cq in ipairs(current or {}) do
    if cq.objectives then
      for _, o in ipairs(cq.objectives) do
        local keyObj = (tostring(cq.title) .. "\n" .. tostring(o.text))
        local prevO = prevObj[keyObj]
        -- Emit only when we see numeric progress change or completion flips
        local curChanged = prevO and o.cur and prevO.cur ~= o.cur
        local completionFlipped = (prevO and prevO.complete ~= o.complete) or (not prevO and o.complete)
        local totalKnown = o.total
        if curChanged or completionFlipped then
          _logEvent("quests", "objective", {
            id = cq.id,
            title = cq.title,
            objective = o.text,
            progress = o.cur,     -- may be nil if the line didn't include numbers
            total    = totalKnown, -- may be nil
            complete = o.complete,
          })
        end
      end
    end
  end

  -- === Persist current snapshot for export AND next diff ===
  char.snapshots = char.snapshots or {}
  char.snapshots.quest_log = { ts = now(), quests = current }
  prev.snapshots.quest_log = current
end

--------------------------------------------------------------------------------
-- Reputation (series snapshot per faction)
--------------------------------------------------------------------------------
function NS.Quests_ScanReputation()
  local key, char = EnsureCharacterBranches()
  char.series.reputation = char.series.reputation or {}

  local num = (type(GetNumFactions)=="function" and GetNumFactions()) or 0
  for i = 1, num do
    local name, _, standingId, barMin, barMax, barValue, _, _, isHeader = GetFactionInfo(i)
    if name and not isHeader then
      push(char.series.reputation, {
        ts = now(), name = name, standing_id = standingId,
        value = barValue, min = barMin, max = barMax
      })
    end
  end
end

--------------------------------------------------------------------------------
-- Skills (snapshot of ranks) + tradeskill learned/changed events
--------------------------------------------------------------------------------
function NS.Quests_ScanSkills()
  local key, char = EnsureCharacterBranches()
  char.series.skills = {}

  local num = (type(GetNumSkillLines)=="function" and GetNumSkillLines()) or 0
  for i = 1, num do
    local name, isHeader, _, rank, _, maxRank = GetSkillLineInfo(i)
    if not isHeader and name then
      table.insert(char.series.skills, { name = name, rank = rank or 0, max = maxRank or 0 })
    end
  end

  -- === DIFF for professions ===
  local prev = _ensurePrevStore(char)
  local oldSkills = (prev.snapshots and prev.snapshots.skills) or {}
  local newSkills = char.series.skills

  -- Consider only profession-like lines.
  local PROF = {
    ["Alchemy"]=true, ["Blacksmithing"]=true, ["Enchanting"]=true, ["Engineering"]=true,
    ["Herbalism"]=true, ["Inscription"]=true, ["Jewelcrafting"]=true, ["Leatherworking"]=true,
    ["Mining"]=true, ["Skinning"]=true, ["Tailoring"]=true, ["Cooking"]=true, ["First Aid"]=true,
    ["Fishing"]=true, ["Riding"]=true,
  }

  local function _filterProf(arr)
    local out = {}
    for _, v in ipairs(arr or {}) do if PROF[v.name] then table.insert(out, v) end end
    return out
  end

  local added, removed, common = _setDiff(_filterProf(oldSkills), _filterProf(newSkills))

  for _, s in ipairs(added) do
    _logEvent("tradeskill", "learned", { skill = s.name, rank = s.rank })
  end

  -- Rank changes
  for name, pair in pairs(common) do
    local oldR = pair.old.rank or 0
    local newR = pair.new.rank or 0
    if newR ~= oldR then
      _logEvent("tradeskill", "changed", { skill = name, rank = newR, from = oldR, max = pair.new.max or pair.old.max })
    end
  end

  -- Persist snapshot for future diffs (kept in prev.snapshots to avoid bloating char.series)
  prev.snapshots.skills = newSkills
end

--------------------------------------------------------------------------------
-- Spellbook (snapshot + learned/unlearned + rank_changed)
--------------------------------------------------------------------------------
function NS.Quests_ScanSpellbook()
  local tabs = {}
  local numTabs = (type(GetNumSpellTabs)=="function" and GetNumSpellTabs()) or 0
  for t = 1, numTabs do
    local name, _, offset, numSpells = GetSpellTabInfo(t)
    local tab = { name = name, spells = {} }
    local count = numSpells or 0
    for i = 1, count do
      local spellName, spellRank = GetSpellName(offset + i, BOOKTYPE_SPELL)
      if spellName then table.insert(tab.spells, { name = spellName, rank = spellRank }) end
    end
    table.insert(tabs, tab)
  end

  local key, char = EnsureCharacterBranches()
  local prev = _ensurePrevStore(char)

  -- DIFF: flatten all spells across tabs by name to avoid double-counting
  local newAll = {}
  for _, tab in ipairs(tabs) do
    for _, s in ipairs(tab.spells or {}) do newAll[#newAll+1] = s end
  end

  local oldAll = {}
  do
    local oldSnap = char.snapshots and char.snapshots.spellbook or nil
    if oldSnap and type(oldSnap.tabs) == "table" then
      for _, tab in ipairs(oldSnap.tabs) do
        for _, s in ipairs(tab.spells or {}) do oldAll[#oldAll+1] = s end
      end
    end
  end

  local added, removed, common = _setDiff(oldAll, newAll)

  for _, s in ipairs(added)   do _logEvent("spellbook", "learned",   { spell = s.name, rank = s.rank }) end
  for _, s in ipairs(removed) do _logEvent("spellbook", "unlearned", { spell = s.name, rank = s.rank }) end

  -- Optional: rank changes
  for name, pair in pairs(common) do
    local oldR = pair.old.rank or ""
    local newR = pair.new.rank or ""
    if oldR ~= newR then
      _logEvent("spellbook", "rank_changed", { spell = name, from = oldR, to = newR })
    end
  end

  -- Save current snapshot
  char.snapshots.spellbook = { ts = now(), tabs = tabs }
  prev.snapshots.spellbook = char.snapshots.spellbook
end

--------------------------------------------------------------------------------
-- Glyphs (snapshot + added/removed/changed)
--------------------------------------------------------------------------------
function NS.Quests_ScanGlyphs()
  local glyphs = {}
  local numSockets = (type(GetNumGlyphSockets)=="function" and GetNumGlyphSockets()) or 0
  for i = 1, numSockets do
    local enabled, gtype, spellID = GetGlyphSocketInfo(i)
    if enabled and spellID then
      local name, _, icon = GetSpellInfo(spellID)
      table.insert(glyphs, { name = name, type = gtype, icon = icon, spell_id = spellID, socket = i })
    end
  end

  local key, char = EnsureCharacterBranches()
  local prev = _ensurePrevStore(char)

  local oldGlyphs = (char.snapshots and char.snapshots.glyphs and char.snapshots.glyphs.glyphs) or {}
  local added, removed, common = _setDiff(oldGlyphs, glyphs)

  for _, g in ipairs(added)   do _logEvent("glyph", "added",   { glyph = g.name, type = g.type, socket = g.socket }) end
  for _, g in ipairs(removed) do _logEvent("glyph", "removed", { glyph = g.name, type = g.type, socket = g.socket }) end

  for name, pair in pairs(common) do
    -- socket/type changes count as "changed"
    if (pair.old.type ~= pair.new.type) or (pair.old.socket ~= pair.new.socket) then
      _logEvent("glyph", "changed", { glyph = name, from = {type=pair.old.type, socket=pair.old.socket}, to = {type=pair.new.type, socket=pair.new.socket} })
    end
  end

  char.snapshots.glyphs = { ts = now(), glyphs = glyphs }
  prev.snapshots.glyphs = char.snapshots.glyphs
end

--------------------------------------------------------------------------------
-- Companions (mounts + critters snapshot)
--------------------------------------------------------------------------------
function NS.Quests_ScanCompanions()
  local mounts, critters = {}, {}

  local function scanType(t, store)
    local n = (type(GetNumCompanions)=="function" and GetNumCompanions(t)) or 0
    for i = 1, n do
      local creatureID, creatureName, spellID, icon, isActive = GetCompanionInfo(t, i)
      push(store, { name = creatureName, icon = icon, creature_id = creatureID, spell_id = spellID, active = isActive, type = t })
    end
  end

  scanType("MOUNT", mounts)
  scanType("CRITTER", critters)

  local key, char = EnsureCharacterBranches()
  char.snapshots.companions = { ts = now(), mount = mounts, critter = critters }
end

--------------------------------------------------------------------------------
-- Stable snapshot
--------------------------------------------------------------------------------
function NS.Quests_ScanPetStable()
  local num = (type(GetNumStableSlots)=="function" and GetNumStableSlots()) or 0
  local stable = {}
  for i = 1, num do
    local hasPet, level, name, icon, family = GetStablePetInfo(i)
    if hasPet then push(stable, { name = name, level = level, icon = icon, family = family, slot = i }) end
  end

  local key, char = EnsureCharacterBranches()
  char.snapshots.pet_stable = { ts = now(), pets = stable }
end

--------------------------------------------------------------------------------
-- Pet info snapshot
--------------------------------------------------------------------------------
function NS.Quests_ScanPetInfo()
  if type(HasPetUI) ~= "function" or not HasPetUI() then return end
  local name   = UnitName("pet")
  local family = UnitCreatureFamily("pet")
  local curXP, nextXP = GetPetExperience()

  local key, char = EnsureCharacterBranches()
  char.snapshots.pet = { ts = now(), name = name, family = family, xp = curXP, next = nextXP }
end

--------------------------------------------------------------------------------
-- Pet spellbook snapshot
--------------------------------------------------------------------------------
function NS.Quests_ScanPetSpellbook()
  if type(HasPetSpells) ~= "function" or not HasPetSpells() then return end
  local spells = {}
  for i = 1, 200 do
    local name = GetSpellName(i, BOOKTYPE_PET)
    if not name then break end
    push(spells, { name = name })
  end

  local key, char = EnsureCharacterBranches()
  char.snapshots.pet_spells = { ts = now(), spells = spells }
end

--------------------------------------------------------------------------------
-- Trade skills (snapshot) + recipe learned events
--------------------------------------------------------------------------------
function NS.Quests_ScanTradeSkills()
  -- Warmane/private cores: full data typically requires the TradeSkill UI open; handle partial gracefully
  local count = (type(GetNumTradeSkills) == "function" and GetNumTradeSkills()) or 0
  local key, char = EnsureCharacterBranches()

  -- If the TradeSkill UI is open, we can also capture the profession header (optional context for events)
  local profName, profRank, profMaxRank = nil, nil, nil
  if type(GetTradeSkillLine) == "function" then
    -- WotLK signature: name, currentRank, maxRank
    profName, profRank, profMaxRank = GetTradeSkillLine()
  end

  -- Build the current snapshot
  char.snapshots.tradeskills = {}
  for i = 1, count do
    local skillName, skillType = GetTradeSkillInfo(i)
    local link  = (type(GetTradeSkillItemLink) == "function" and GetTradeSkillItemLink(i)) or nil
    local icon  = (type(GetTradeSkillIcon) == "function" and GetTradeSkillIcon(i)) or nil

    local numMadeMin, numMadeMax = nil, nil
    if type(GetTradeSkillNumMade) == "function" then
      numMadeMin, numMadeMax = GetTradeSkillNumMade(i)
    end

    local reagents = {}
    local nReagents = (type(GetTradeSkillNumReagents) == "function" and GetTradeSkillNumReagents(i)) or 0
    for r = 1, nReagents do
      local rName, rTexture, rCount, rHave = GetTradeSkillReagentInfo(i, r)
      local rLink = (type(GetTradeSkillReagentItemLink) == "function" and GetTradeSkillReagentItemLink(i, r)) or nil
      push(reagents, { name = rName, count = rCount or 0, have = rHave or 0, icon = rTexture, link = rLink })
    end

    local cooldown = (type(GetTradeSkillCooldown) == "function" and GetTradeSkillCooldown(i)) or 0
    local cooldownText = (type(SecondsToTime) == "function" and cooldown > 0 and SecondsToTime(cooldown)) or nil

    push(char.snapshots.tradeskills, {
      profession    = profName,
      name          = skillName,
      type          = skillType,
      link          = link,
      icon          = icon,
      num_made_min  = numMadeMin,
      num_made_max  = numMadeMax,
      reagents      = reagents,
      cooldown      = cooldown,
      cooldown_text = cooldownText,
    })
  end

  -- Emit events for newly learned recipes by diffing previous vs current snapshot.
  -- prev is rooted in WhoDatDB (a SavedVariable) so it persists across reloads/relogs,
  -- meaning only genuinely new recipes will fire recipe_learned events.
  do
    WhoDatDB.characters[key].prev = WhoDatDB.characters[key].prev or {}
    WhoDatDB.characters[key].prev.snapshots = WhoDatDB.characters[key].prev.snapshots or {}
    local prev = WhoDatDB.characters[key].prev

    local oldRows = prev.snapshots.tradeskills or {}
    local newRows = char.snapshots.tradeskills or {}

    local added, removed = _setDiff(oldRows, newRows)

    for _, row in ipairs(added) do
      _logEvent("tradeskill", "recipe_learned", {
        recipe          = row.name,
        link            = row.link,
        icon            = row.icon,
        profession      = profName,
        profession_rank = profRank,
        profession_max  = profMaxRank,
      })
    end

    -- Persist the current snapshot for future diffs
    prev.snapshots.tradeskills = newRows
  end

  -- Stamp
  char.snapshots.tradeskills_ts = now()
end

--------------------------------------------------------------------------------
-- Optional: event registration for modules that prefer self-wiring
-- Safe to call multiple times.
--------------------------------------------------------------------------------
function NS.Quests_RegisterEvents()
  if NS._questsFrame then return end
  local f = CreateFrame("Frame")
  NS._questsFrame = f

  f:SetScript("OnEvent", function(self, event)
    if event == "UPDATE_FACTION" then
      safe(NS.Quests_ScanReputation)

    elseif event == "GLYPH_ADDED" or event == "GLYPH_REMOVED" or event == "GLYPH_UPDATED" then
      safe(NS.Quests_ScanGlyphs)

    elseif event == "COMPANION_UPDATE" or event == "COMPANION_LEARNED" then
      safe(NS.Quests_ScanCompanions)

    elseif event == "QUEST_LOG_UPDATE" then
      safe(NS.Quests_ScanLog)

    -- Fast-path emit on QUEST_COMPLETE
    -- QUEST_COMPLETE fires after turning in; the quest may soon disappear from the log.
    elseif event == "QUEST_COMPLETE" then
      safe(function()
        -- Ensure character branch + prev store
        local key, char = EnsureCharacterBranches()
        char.prev = _ensurePrevStore(char)

        -- If available, use the current selection to read the completed quest
        local sel = type(GetQuestLogSelection) == "function" and GetQuestLogSelection() or nil
        if sel and type(GetQuestLogTitle) == "function" then
          local title, level, _, isHeader, _, isComplete, questID = GetQuestLogTitle(sel)
          
          -- FIXED: Use index as ID if questID is nil (for servers that don't return questID)
          local useID = questID or sel
          
          -- Emit immediately if we have identifiers (don't require questID anymore)
          if title and not isHeader and useID then
            markCompleted(useID, questRewardPending.quest_title)
            _logEvent("quests", "completed", { id = useID, title = title })
          end
        end
      end)

      -- Capture quest reward info
      safe(OnQuestCompleteDialog)

      -- Refresh the log after completion (may remove the quest or update state)
      safe(NS.Quests_ScanLog)

    elseif event == "QUEST_FINISHED" then
      -- NEW: Mark quest as completed using reward data as backup
      -- (in case QUEST_COMPLETE doesn't fire on this server)
      safe(function()
        if questRewardPending and (questRewardPending.quest_id or questRewardPending.quest_title) then
          local key, char = EnsureCharacterBranches()
          char.prev = _ensurePrevStore(char)
          
          local useID = questRewardPending.quest_id
          
          -- If no quest_id, try to find in log by title before it disappears
          if not useID and type(GetNumQuestLogEntries) == "function" and type(GetQuestLogTitle) == "function" then
            local numEntries = GetNumQuestLogEntries()
            for i = 1, numEntries do
              local title, _, _, isHeader, _, _, questID = GetQuestLogTitle(i)
              if not isHeader and title == questRewardPending.quest_title then
                useID = questID or i
                break
              end
            end
          end
          
          -- Fallback: use title hash as ID
          if not useID and questRewardPending.quest_title then
            local hash = 0
            for i = 1, #questRewardPending.quest_title do
              hash = hash + string.byte(questRewardPending.quest_title, i)
            end
            useID = hash
          end
          
          if useID then
            markCompleted(useID, questRewardPending.quest_title)
            _logEvent("quests", "completed", { id = useID, title = questRewardPending.quest_title })
            
            if NS.Log then
              NS.Log("INFO", "Quest marked COMPLETED via QUEST_FINISHED: %s (ID: %s)", 
                questRewardPending.quest_title, tostring(useID))
            end
          end
        end
      end)
      
      -- Original: Save quest reward data
      safe(OnQuestTurnedIn)

    elseif event == "TRADE_SKILL_UPDATE" then
      safe(NS.Quests_ScanTradeSkills)

    elseif event == "PLAYER_LOGIN" then
  -- Detect quest addons
  DetectQuestAddons()
  
  if activeQuestSource ~= "WhoDAT_builtin" then
    print(string.format("[WhoDAT] Quest tracking enhanced with: %s", activeQuestSource))
  end
  
elseif event == "PLAYER_ENTERING_WORLD" then
  safe(NS.Quests_ScanAll)
    end
  end)

  -- Event subscriptions
  f:RegisterEvent("PLAYER_ENTERING_WORLD")
  f:RegisterEvent("UPDATE_FACTION")
  f:RegisterEvent("GLYPH_ADDED")
  f:RegisterEvent("GLYPH_REMOVED")
  f:RegisterEvent("GLYPH_UPDATED")
  f:RegisterEvent("COMPANION_UPDATE")
  f:RegisterEvent("COMPANION_LEARNED")
  f:RegisterEvent("QUEST_LOG_UPDATE")
  f:RegisterEvent("QUEST_COMPLETE") -- required for fast-path emit
  f:RegisterEvent("QUEST_FINISHED") -- quest actually turned in
  f:RegisterEvent("TRADE_SKILL_UPDATE")
  f:RegisterEvent("PLAYER_LOGIN")
end

-- Auto-wire on load (safe: function guards against double-registration)
NS.Quests_RegisterEvents()